/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

/* error 0x1 = 1st bit in binary, indicates an unspecified error.
 * error 0x2 = 2nd bit in binary, indicates high temperature in the PA.
 * error 0x4 = 3rd bit in binary, indicates very high temperature in the PA, resulting in a shutdown (?).
 * error 0x8 = 4th bit in binary, indicates high reflection.
 * error 0x10 = 5th bit in binary, indicates very high reflection, resulting in a shutdown (?).
 * error 0x20 = 6th bit in binary, indicates a reset has occured.
 * error 0x40 = 7th bit in binary, indicates a temperature read error.
 * error 0x80 = 8th bit in binary, indicates a power measurement failure.
 * error 0x100 = 9th bit in binary, indicates a failure to enable the RF power output.
 * error 0x200 = 10th bit in binary, indicates multiplexer failure
 * error 0x400 = 11th bit in binary, indicates that the external shutdown has been triggered (pin 8 on the D-sub).
 * error 0x800 = 12th bit in binary, indicates the SG board has run out of memory.
 * error 0x1000 = 13th bit in binary, indicates an I2C communcation problem.
 * error 0x2000 = 14th bit in binary, indicates an SPI communcation problem.
 */

#include <QFile>
#include "statuscheck.h"
#include "serial_v2.h"

//
// TODO:
// Convert Old GUI to use the newer StatusCheck stuff and delete all the legacy functions.
// Remove all old serial_v2 stuff.
// When legacy StatusCheck is gone, the new architecture GUIs can also stop compiling in serial_v2.
//

StatusCheck::StatusCheck(QSerialPort *SG_port, QString channel_select, QString messages_file, bool console_output_enabled)
{
	this->SG_port = SG_port;
	this->console_output_enabled = console_output_enabled;
	this->channel_select = channel_select;

	QFile file(messages_file);
	if(!file.open(QIODevice::ReadOnly))
	{
		qDebug() << "Messages File unavailable!" << file.errorString() << "\n" << file.fileName();

		error_map.insert("none", "OK");
		error_map.insert("0", "Unspecified error.");
		error_map.insert("1", "High temperature.");
		error_map.insert("2", "Shutdown temperature.");
		error_map.insert("3", "High reflection.");
		error_map.insert("4", "Shutdown reflection.");
		error_map.insert("5", "Reset detected.");
		error_map.insert("6", "Temperature measurement failure.");
		error_map.insert("7", "Power measurement failure.");
		error_map.insert("8", "RF enable failure.");
		error_map.insert("9", "Multiplexer failure.");
		error_map.insert("10", "External shutdown detected.");
		error_map.insert("11", "Out of memory.");
		error_map.insert("12", "I2C error.");
		error_map.insert("13", "SPI error.");
		error_map.insert("14", "IQ conversion error.");
		error_map.insert("15", "SOA measurement error.");
		error_map.insert("16", "External watchdog timeout.");
		error_map.insert("17", "Calibration missing.");
		error_map.insert("18", "External protection triggered.");
		error_map.insert("19", "SOA high dissipation.");
		error_map.insert("20", "SOA shutdown dissipation.");
		error_map.insert("21", "Calibration EEPROM outdated.");
		error_map.insert("22", "PA error.");
		error_map.insert("23", "PA reset failure.");
		error_map.insert("24", "PA High current.");
		error_map.insert("25", "IQ demodulator lock lost.");
		error_map.insert("26", "Alarm IN signaled.");
		error_map.insert("27", "IQ modulator lock lost.");
		error_map.insert("28", "SOA high current.");
		error_map.insert("29", "SOA shutdown current.");
		error_map.insert("30", "SOA high forward power..");
		error_map.insert("31", "SOA shutdown forward power.");
		error_map.insert("32", "SOA shutdown minimum voltage.");
		error_map.insert("33", "SOA low voltage.");
		error_map.insert("34", "SOA high voltage.");
		error_map.insert("35", "SOA shutdown maxmimum voltage.");
		error_map.insert("else", "Unknown error.");
	}
	else
	{
		/* Read the file and save values to configmap */
		while(!file.atEnd())
		{
			QString line;
			line = file.readLine();
			line.replace("\t","");
			line.replace("\r","");
			line.replace("\n","");

			if (!(line.startsWith("#") || line.isEmpty()))
			{
				 QStringList list = line.split(QRegExp("\\s*=\\s*"));
				 if(list.count() == 2)
				 {
					error_map.insert(list.at(0), list.at(1));
				 }
				 else
				 {
					qDebug() << "line syntax incorrect: " << list << "ignored line";
				 }
			}
		}
	}
	file.close();
}


//
// TODO:
// Remove legacy content when no other code relies on this.
//
void StatusCheck::ST_parser()
{
	QString get_string = serial_getStatus(*SG_port, channel_select, console_output_enabled);
	QRegExp regexp;
	regexp.setCaseSensitivity(Qt::CaseInsensitive);
	regexp.setPattern("\\$ST,\\d+,(\\S+),(\\S+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		last_error = error;
		status = regexp.cap(1).toULongLong(nullptr,16);
		error = regexp.cap(2).toULongLong(nullptr,16);

		if (console_output_enabled == true)
		{
			qDebug() << "SG Status: " << QString("0x%1").arg(status,0,16);
			qDebug() << "SG Error: " << QString("0x%1").arg(error,0,16) << "\n";
		}
	}
}

//
// TODO:
// Remove legacy content when no other code relies on this.
//
bool StatusCheck::ECG_parser()
{
	QString get_string = serial_getRF_status(*SG_port,channel_select, console_output_enabled);
	QRegExp regexp("\\$ECG,\\d+,(\\d+)\r\n");

	if (regexp.indexIn(get_string)>=0)
	{
		RF_status = (bool)regexp.cap(1).toInt();
		if (console_output_enabled == true)
		{
			qDebug() << "RF_enabled" << RF_status << "\n";
		}
	}
	return RF_status;
}


//
// TODO:
// Remove legacy content when no other code relies on this.
//
/* Check if error state has changed, Emit signal appropriate signal and information if that's the case. */
void StatusCheck::poll_SG_error()
{
	ST_parser();
	if (last_error != error)
	{
		emit SG_error_signal(&error);
	}

	/* If there's an error:
	 * 1. Check for a reset and deal with it.
	 * 2. Poll RF enable to see if SOA is shutting down the RF output and deal with it.
	 * 3. Clear the error */
	if (error > 0)
	{
		if (error & (1<<5))
		{
			emit reset_detected_signal();
		}
		if (ECG_parser() == false)
		{
			emit RF_disabled_signal();
		}

		clear_error();
	}
}

//
// TODO:
// Remove legacy content when no other code relies on this.
//
void StatusCheck::clear_error()
{
	serial_clearError(*SG_port, channel_select, console_output_enabled);
}

/* New Code; keep this */
StatusCheck::StatusCheck(Serial *port, QString messages_file)
{
	//
	// TODO:
	// SGX_port is never used. Delete!
	//
	SGX_port = port;

	QFile file(messages_file);
	if(!file.open(QIODevice::ReadOnly))
	{
		qDebug() << "Messages File unavailable!" << file.errorString() << "\n" << file.fileName();

		error_map.insert("none", "OK");
		error_map.insert("0", "Unspecified error.");
		error_map.insert("1", "High temperature.");
		error_map.insert("2", "Shutdown temperature.");
		error_map.insert("3", "High reflection.");
		error_map.insert("4", "Shutdown reflection.");
		error_map.insert("5", "Reset detected.");
		error_map.insert("6", "Temperature measurement failure.");
		error_map.insert("7", "Power measurement failure.");
		error_map.insert("8", "RF enable failure.");
		error_map.insert("9", "Multiplexer failure.");
		error_map.insert("10", "External shutdown detected.");
		error_map.insert("11", "Out of memory.");
		error_map.insert("12", "I2C error.");
		error_map.insert("13", "SPI error.");
		error_map.insert("14", "IQ conversion error.");
		error_map.insert("15", "SOA measurement error.");
		error_map.insert("16", "External watchdog timeout.");
		error_map.insert("17", "Calibration missing.");
		error_map.insert("18", "External protection triggered.");
		error_map.insert("19", "SOA high dissipation.");
		error_map.insert("20", "SOA shutdown dissipation.");
		error_map.insert("21", "Calibration EEPROM outdated.");
		error_map.insert("22", "PA error.");
		error_map.insert("23", "PA reset failure.");
		error_map.insert("24", "PA High current.");
		error_map.insert("25", "IQ demodulator lock lost.");
		error_map.insert("26", "Alarm IN signaled.");
		error_map.insert("27", "IQ modulator lock lost.");
		error_map.insert("28", "SOA high current.");
		error_map.insert("29", "SOA shutdown current.");
		error_map.insert("30", "SOA high forward power..");
		error_map.insert("31", "SOA shutdown forward power.");
		error_map.insert("32", "SOA shutdown minimum voltage.");
		error_map.insert("33", "SOA low voltage.");
		error_map.insert("34", "SOA high voltage.");
		error_map.insert("35", "SOA shutdown maxmimum voltage.");
		error_map.insert("else", "Unknown error.");
	}
	else
	{
		/* Read the file and save values to configmap */
		while(!file.atEnd())
		{
			QString line;
			line = file.readLine();
			line.replace("\t","");
			line.replace("\r","");
			line.replace("\n","");

			if (!(line.startsWith("#") || line.isEmpty()))
			{
				 QStringList list = line.split(QRegExp("\\s*=\\s*"));
				 if(list.count() == 2)
				 {
					error_map.insert(list.at(0), list.at(1));
				 }
				 else
				 {
					qDebug() << "line syntax incorrect: " << list << "ignored line";
				 }
			}
		}
	}
	file.close();
}

QStringList StatusCheck::translate_SG_error(quint64 error)
{
	QStringList error_message_list;

	if (error > 0)
	{
		/* 64-bit error check */
		for (int i = 0; i < 64; i++)
		{
			/* The '1' must be cast to a 64bit unsigned int / unsigned long long
			 * Otherwise the bitwise AND-operation goes wrong */
			if (error & ((quint64) 1<<i))
			{
				if (error_map.contains(QString::number(i)))
				{
					error_message_list.append(error_map.value(QString::number(i)));
				}
				else if (error_map.contains("else"))
				{
					error_message_list.append(error_map.value("else"));
				}
				else
				{
					error_message_list.append("Unknown Error");
				}
			}
		}
	}
	else
	{
		if (error_map.contains("none"))
		{
			error_message_list.append(error_map.value("none"));
		}
		else
		{
			error_message_list.append("");
		}
	}

	return error_message_list;
}
